<?php
// index.php - The slideshow page

// Load slides from JSON file
$slidesFile = 'bijwerken/slides.json';
$slides = [];
if (file_exists($slidesFile)) {
    $slides = json_decode(file_get_contents($slidesFile), true);
}

// Function to check if a slide is active based on current time and schedule
function isSlideActive($slide) {
    $now = new DateTime();
    $start = DateTime::createFromFormat('Y-m-d H:i', $slide['start']);
    $end = DateTime::createFromFormat('Y-m-d H:i', $slide['end']);

    if (!$start || !$end) return false;

    // Basic check: after start and before end
    if ($now < $start || $now > $end) return false;

    // Handle repeats
    if (isset($slide['repeat_type']) && $slide['repeat_type'] !== 'none') {
        $interval = isset($slide['interval']) ? (int)$slide['interval'] : 1;

        switch ($slide['repeat_type']) {
            case 'daily':
            case 'every_x_days':
                // Calculate days since start
                $daysSinceStart = $start->diff($now)->days;
                if ($daysSinceStart % $interval !== 0) return false;
                break;

            case 'weekly':
            case 'every_x_weeks':
                // Weeks since start
                $weeksSinceStart = floor($start->diff($now)->days / 7);
                if ($weeksSinceStart % $interval !== 0) return false;
                break;

            case 'monthly':
            case 'every_x_months':
                // Months since start
                $monthsSinceStart = ($now->format('Y') - $start->format('Y')) * 12 + ($now->format('m') - $start->format('m'));
                if ($monthsSinceStart % $interval !== 0) return false;
                break;

            case 'yearly':
            case 'every_x_years':
                // Years since start
                $yearsSinceStart = $now->format('Y') - $start->format('Y');
                if ($yearsSinceStart % $interval !== 0) return false;
                break;
        }
    }

    return true;
}

// Filter active slides
$activeSlides = array_filter($slides, 'isSlideActive');
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="utf-8">
    <meta http-equiv="X-UA-Compatible" content="IE=edge,chrome=1">
    <meta http-equiv="refresh" content="900; URL=index.php">
    <meta http-equiv="expires" content="timestamp">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Beeldkrant Slideshow</title>
    <style>
        html, body {
            margin: 0;
            padding: 0;
            overflow: hidden;
            height: 100%;
            width: 100%;
        }
        .slideshow {
            position: relative;
            height: 100%;
            width: 100%;
        }
        .slide {
            position: absolute;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            opacity: 0;
            transition: opacity 1s ease-in-out;
            pointer-events: none;
        }
        .slide.active {
            opacity: 1;
            pointer-events: auto;
        }
        .background {
            width: 100%;
            height: 100%;
            object-fit: cover;
            position: absolute;
            top: 0;
            left: 0;
        }
        video.background {
            controls: false;
        }
        .logo {
            position: absolute;
            top: 20px;
            left: 20px;
            z-index: 10;
            width: 150px;
            height: auto;
        }
    </style>
</head>
<body>
    <div class="slideshow">
        <?php foreach ($activeSlides as $index => $slide): ?>
            <div class="slide" data-type="<?php echo htmlspecialchars($slide['type']); ?>" <?php if ($slide['type'] === 'image') echo 'data-duration="' . htmlspecialchars($slide['duration']) . '"'; ?>>
                <?php if ($slide['type'] === 'image'): ?>
                    <img class="background" src="bijwerken/<?php echo htmlspecialchars($slide['file']); ?>" alt="<?php echo htmlspecialchars($slide['name']); ?>">
                <?php elseif ($slide['type'] === 'video'): ?>
                    <video class="background" src="bijwerken/<?php echo htmlspecialchars($slide['file']); ?>" muted autoplay></video>
                <?php endif; ?>
                <?php if (!empty($slide['logo'])): ?>
                    <img class="logo" src="bijwerken/<?php echo htmlspecialchars($slide['logo']); ?>" alt="Logo for <?php echo htmlspecialchars($slide['name']); ?>">
                <?php endif; ?>
            </div>
        <?php endforeach; ?>
    </div>

    <script>
        const slides = document.querySelectorAll('.slide');
        let currentIndex = 0;
        let transitionTimeout;

        function showNextSlide() {
            if (slides[currentIndex]) {
                slides[currentIndex].classList.remove('active');
                const video = slides[currentIndex].querySelector('video');
                if (video) {
                    video.pause();
                    video.currentTime = 0;
                }
            }

            currentIndex = (currentIndex + 1) % slides.length;

            const currentSlide = slides[currentIndex];
            currentSlide.classList.add('active');

            const type = currentSlide.getAttribute('data-type');

            if (type === 'image') {
                const duration = parseInt(currentSlide.getAttribute('data-duration')) || 5000;
                clearTimeout(transitionTimeout);
                transitionTimeout = setTimeout(showNextSlide, duration);
            } else if (type === 'video') {
                const video = currentSlide.querySelector('video');
                if (video) {
                    video.currentTime = 0;
                    video.play();
                    video.onended = showNextSlide;
                } else {
                    clearTimeout(transitionTimeout);
                    transitionTimeout = setTimeout(showNextSlide, 5000);
                }
            }
        }

        if (slides.length > 0) {
            showNextSlide();
        }
    </script>
</body>
</html>
