<?php
// Nederlands: Beheerpagina voor het toevoegen, bewerken en verwijderen van slides
// English: Admin page for adding, editing, and deleting slides
$slidesFile = 'slides.json';
$uploadsDir = 'uploads/';
$imagesDir = $uploadsDir . 'images/';
$videosDir = $uploadsDir . 'videos/';
$logosDir = $uploadsDir . 'logos/';

// Nederlands: Maak mappen aan als ze niet bestaan en controleer schrijfrechten
// English: Create directories if they don't exist and check write permissions
foreach ([$imagesDir, $videosDir, $logosDir] as $dir) {
    if (!is_dir($dir)) {
        if (!mkdir($dir, 0777, true)) {
            die("Fout: Kan map $dir niet aanmaken. Controleer serverrechten. / Error: Cannot create directory $dir. Check server permissions.");
        }
    }
    if (!is_writable($dir)) {
        die("Fout: Map $dir is niet beschrijfbaar. Stel rechten in (bijv. chmod 777 $dir). / Error: Directory $dir is not writable. Set permissions (e.g., chmod 777 $dir).");
    }
}

// Nederlands: Zorg ervoor dat slides.json bestaat en beschrijfbaar is
// English: Ensure slides.json exists and is writable
if (!file_exists($slidesFile)) {
    file_put_contents($slidesFile, json_encode([]));
}
if (!is_writable($slidesFile)) {
    die("Fout: slides.json is niet beschrijfbaar. Stel rechten in (bijv. chmod 666 slides.json). / Error: slides.json is not writable. Set permissions (e.g., chmod 666 slides.json).");
}

// Nederlands: Laad slides uit slides.json
// English: Load slides from slides.json
$slides = file_exists($slidesFile) ? json_decode(file_get_contents($slidesFile), true) : [];
if ($slides === null || !is_array($slides)) {
    error_log("Fout: Kan slides.json niet decoderen of ongeldige data. Bestand wordt gereset. / Error: Cannot decode slides.json or invalid data. Resetting file.");
    $slides = [];
}
$errors = [];

// Nederlands: Vertalingstabel voor foutmeldingen
// English: Translation table for error messages
$errorMessages = [
    'name_required' => ['nl' => 'Naam is verplicht.', 'en' => 'Name is required.'],
    'invalid_start_date' => ['nl' => 'Startdatum/tijd moet in formaat YYYY-MM-DD HH:MM zijn.', 'en' => 'Start date/time must be in YYYY-MM-DD HH:MM format.'],
    'invalid_end_date' => ['nl' => 'Einddatum/tijd moet in formaat YYYY-MM-DD HH:MM zijn.', 'en' => 'End date/time must be in YYYY-MM-DD HH:MM format.'],
    'end_before_start' => ['nl' => 'Einddatum/tijd moet na de startdatum/tijd liggen.', 'en' => 'End date/time must be after start date/time.'],
    'invalid_interval' => ['nl' => 'Interval moet minimaal 1 zijn.', 'en' => 'Interval must be at least 1.'],
    'invalid_duration' => ['nl' => 'Duurtijd moet minimaal 1000ms zijn voor afbeeldingen.', 'en' => 'Duration must be at least 1000ms for images.'],
    'file_required' => ['nl' => 'Een bestand is verplicht bij het toevoegen van een slide.', 'en' => 'A file is required when adding a slide.'],
    'invalid_file_type' => [
        'nl' => ['image' => 'Alleen JPG/PNG/WEBP/GIF toegestaan voor afbeeldingen.', 'video' => 'Alleen MP4/WEBM toegestaan voor video’s.', 'logo' => 'Alleen JPG/PNG/WEBP/GIF toegestaan voor logo’s.'],
        'en' => ['image' => 'Only JPG/PNG/WEBP/GIF allowed for images.', 'video' => 'Only MP4/WEBM allowed for videos.', 'logo' => 'Only JPG/PNG/WEBP/GIF allowed for logos.']
    ],
    'file_too_large' => ['nl' => 'Bestand is te groot (max. 4GB).', 'en' => 'File is too large (max. 4GB).'],
    'logo_too_large' => ['nl' => 'Logo is te groot (max. 50MB).', 'en' => 'Logo is too large (max. 50MB).'],
    'upload_failed' => ['nl' => 'Fout bij het uploaden van bestand: Kan niet naar map verplaatsen.', 'en' => 'Error uploading file: Cannot move to directory.'],
    'logo_upload_failed' => ['nl' => 'Fout bij het uploaden van logo: Kan niet naar map verplaatsen.', 'en' => 'Error uploading logo: Cannot move to directory.'],
    'file_size_limit' => ['nl' => 'Bestand is te groot voor serverlimiet. Pas upload_max_filesize aan in php.ini.', 'en' => 'File is too large for server limit. Adjust upload_max_filesize in php.ini.'],
    'logo_size_limit' => ['nl' => 'Logo is te groot voor serverlimiet. Pas upload_max_filesize aan in php.ini.', 'en' => 'Logo is too large for server limit. Adjust upload_max_filesize in php.ini.'],
    'upload_error' => ['nl' => 'Fout bij het uploaden van bestand: Onbekende fout (code %s).', 'en' => 'Error uploading file: Unknown error (code %s).'],
    'logo_upload_error' => ['nl' => 'Fout bij het uploaden van logo: Onbekende fout (code %s).', 'en' => 'Error uploading logo: Unknown error (code %s).'],
    'invalid_index' => ['nl' => 'Ongeldige slide-index.', 'en' => 'Invalid slide index.'],
    'delete_failed' => ['nl' => 'Kan slide niet verwijderen: ongeldige index.', 'en' => 'Cannot delete slide: invalid index.'],
    'move_failed' => ['nl' => 'Kan slide niet verplaatsen: ongeldige actie of index.', 'en' => 'Cannot move slide: invalid action or index.'],
    'json_error' => ['nl' => 'Fout bij het genereren van JSON: %s.', 'en' => 'Error generating JSON: %s.'],
    'save_error' => ['nl' => 'Fout bij het opslaan van slides.json. Controleer bestandsrechten.', 'en' => 'Error saving slides.json. Check file permissions.'],
    'remove_file_without_new' => ['nl' => 'Kan bestand niet verwijderen zonder een nieuw bestand te uploaden.', 'en' => 'Cannot remove file without uploading a new one.']
];

// Nederlands: Genereer een unieke bestandsnaam
// English: Generate a unique filename
function generateUniqueFilename($originalName) {
    $ext = pathinfo($originalName, PATHINFO_EXTENSION);
    $baseName = pathinfo($originalName, PATHINFO_FILENAME);
    $uniqueId = time() . '_' . bin2hex(random_bytes(4)); // Timestamp + random string
    return $baseName . '_' . $uniqueId . '.' . $ext;
}

// Nederlands: Valideer bestandstypen
// English: Validate file types
function validateFileType($file, $type) {
    $allowedTypes = [
        'image' => ['image/jpeg', 'image/png', 'image/webp', 'image/gif'],
        'video' => ['video/mp4', 'video/webm'],
        'logo' => ['image/jpeg', 'image/png', 'image/webp', 'image/gif'],
    ];
    return in_array($file['type'], $allowedTypes[$type]);
}

// Nederlands: Valideer datumformaat
// English: Validate date format
function validateDateTime($date, $format = 'Y-m-d H:i') {
    $d = DateTime::createFromFormat($format, $date);
    return $d && $d->format($format) === $date;
}

// Nederlands: Sla slides op in slides.json
// English: Save slides to slides.json
function saveSlides($slides, $file, $lang = 'nl') {
    $json = json_encode($slides, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES);
    if (json_last_error() !== JSON_ERROR_NONE) {
        error_log("JSON encode error: " . json_last_error_msg());
        return sprintf($GLOBALS['errorMessages']['json_error'][$lang], json_last_error_msg());
    }
    if (!file_put_contents($file, $json)) {
        error_log("Fout bij het schrijven naar $file / Error writing to $file");
        return $GLOBALS['errorMessages']['save_error'][$lang];
    }
    error_log("Slides opgeslagen: " . print_r($slides, true));
    return null;
}

// Nederlands: Verwerk formulierinzendingen
// English: Handle form submissions
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action'])) {
    $action = $_POST['action'];
    $lang = isset($_POST['lang']) ? $_POST['lang'] : 'nl';
    if ($action === 'add') {
        if (empty($_POST['name'])) $errors[] = $errorMessages['name_required'][$lang];
        if (!validateDateTime($_POST['start'])) $errors[] = $errorMessages['invalid_start_date'][$lang];
        if (!validateDateTime($_POST['end'])) $errors[] = $errorMessages['invalid_end_date'][$lang];
        if (empty($errors)) {
            $start = DateTime::createFromFormat('Y-m-d H:i', $_POST['start']);
            $end = DateTime::createFromFormat('Y-m-d H:i', $_POST['end']);
            if ($end <= $start) $errors[] = $errorMessages['end_before_start'][$lang];
        }
        if ($_POST['interval'] < 1) $errors[] = $errorMessages['invalid_interval'][$lang];
        if ($_POST['type'] === 'image' && $_POST['duration'] < 1000) $errors[] = $errorMessages['invalid_duration'][$lang];
        if (empty($errors)) {
            $slide = [
                'name' => $_POST['name'],
                'type' => $_POST['type'],
                'start' => $_POST['start'],
                'end' => $_POST['end'],
                'repeat_type' => $_POST['repeat_type'],
                'interval' => (int)$_POST['interval'],
                'file' => '',
                'logo' => '',
            ];
            if ($slide['type'] === 'image') $slide['duration'] = (int)$_POST['duration'];
            if (isset($_FILES['file']) && $_FILES['file']['error'] === UPLOAD_ERR_OK) {
                if (!validateFileType($_FILES['file'], $slide['type'])) {
                    $errors[] = $errorMessages['invalid_file_type'][$lang][$slide['type']];
                } elseif ($_FILES['file']['size'] > 4000000000) {
                    $errors[] = $errorMessages['file_too_large'][$lang];
                } else {
                    $targetDir = $slide['type'] === 'image' ? $imagesDir : $videosDir;
                    $uniqueName = generateUniqueFilename($_FILES['file']['name']);
                    $targetFile = $targetDir . $uniqueName;
                    if (!move_uploaded_file($_FILES['file']['tmp_name'], $targetFile)) {
                        $errors[] = $errorMessages['upload_failed'][$lang];
                        error_log("Upload mislukt: Kan bestand niet verplaatsen naar $targetFile");
                    } else {
                        $slide['file'] = $targetFile;
                        error_log("Bestand geüpload: $targetFile");
                    }
                }
            } else {
                $errors[] = $errorMessages['file_required'][$lang];
                if ($_FILES['file']['error'] === UPLOAD_ERR_INI_SIZE) $errors[] = $errorMessages['file_size_limit'][$lang];
                elseif ($_FILES['file']['error'] !== UPLOAD_ERR_NO_FILE) $errors[] = sprintf($errorMessages['upload_error'][$lang], $_FILES['file']['error']);
            }
            if (isset($_FILES['logo']) && $_FILES['logo']['error'] === UPLOAD_ERR_OK) {
                if (!validateFileType($_FILES['logo'], 'logo')) {
                    $errors[] = $errorMessages['invalid_file_type'][$lang]['logo'];
                } elseif ($_FILES['logo']['size'] > 50000000) {
                    $errors[] = $errorMessages['logo_too_large'][$lang];
                } else {
                    $uniqueName = generateUniqueFilename($_FILES['logo']['name']);
                    $targetFile = $logosDir . $uniqueName;
                    if (!move_uploaded_file($_FILES['logo']['tmp_name'], $targetFile)) {
                        $errors[] = $errorMessages['logo_upload_failed'][$lang];
                        error_log("Logo upload mislukt: Kan bestand niet verplaatsen naar $targetFile");
                    } else {
                        $slide['logo'] = $targetFile;
                        error_log("Logo geüpload: $targetFile");
                    }
                }
            } elseif ($_FILES['logo']['error'] === UPLOAD_ERR_INI_SIZE) {
                $errors[] = $errorMessages['logo_size_limit'][$lang];
            } elseif ($_FILES['logo']['error'] !== UPLOAD_ERR_NO_FILE) {
                $errors[] = sprintf($errorMessages['logo_upload_error'][$lang], $_FILES['logo']['error']);
            }
            if (empty($errors)) {
                $slides[] = $slide;
                error_log("Nieuwe slide toegevoegd: " . print_r($slide, true));
            }
        }
    } elseif ($action === 'edit') {
        $index = (int)$_POST['index'];
        if (!isset($slides[$index])) {
            $errors[] = $errorMessages['invalid_index'][$lang];
            error_log("Edit mislukt: Ongeldige index $index");
        } else {
            if (empty($_POST['name'])) $errors[] = $errorMessages['name_required'][$lang];
            if (!validateDateTime($_POST['start'])) $errors[] = $errorMessages['invalid_start_date'][$lang];
            if (!validateDateTime($_POST['end'])) $errors[] = $errorMessages['invalid_end_date'][$lang];
            if (empty($errors)) {
                $start = DateTime::createFromFormat('Y-m-d H:i', $_POST['start']);
                $end = DateTime::createFromFormat('Y-m-d H:i', $_POST['end']);
                if ($end <= $start) $errors[] = $errorMessages['end_before_start'][$lang];
            }
            if ($_POST['interval'] < 1) $errors[] = $errorMessages['invalid_interval'][$lang];
            if ($_POST['type'] === 'image' && $_POST['duration'] < 1000) $errors[] = $errorMessages['invalid_duration'][$lang];
            if (isset($_POST['remove_file']) && $_POST['remove_file'] === '1' && (!isset($_FILES['file']) || $_FILES['file']['error'] !== UPLOAD_ERR_OK)) {
                $errors[] = $errorMessages['remove_file_without_new'][$lang];
            }
            if (empty($errors)) {
                $slide = [
                    'name' => $_POST['name'],
                    'type' => $_POST['type'],
                    'start' => $_POST['start'],
                    'end' => $_POST['end'],
                    'repeat_type' => $_POST['repeat_type'],
                    'interval' => (int)$_POST['interval'],
                    'file' => $slides[$index]['file'],
                    'logo' => isset($slides[$index]['logo']) ? $slides[$index]['logo'] : '',
                ];
                if ($slide['type'] === 'image') $slide['duration'] = (int)$_POST['duration'];
                if (isset($_FILES['file']) && $_FILES['file']['error'] === UPLOAD_ERR_OK) {
                    if (!validateFileType($_FILES['file'], $slide['type'])) {
                        $errors[] = $errorMessages['invalid_file_type'][$lang][$slide['type']];
                    } elseif ($_FILES['file']['size'] > 4000000000) {
                        $errors[] = $errorMessages['file_too_large'][$lang];
                    } else {
                        $targetDir = $slide['type'] === 'image' ? $imagesDir : $videosDir;
                        $uniqueName = generateUniqueFilename($_FILES['file']['name']);
                        $targetFile = $targetDir . $uniqueName;
                        if (!move_uploaded_file($_FILES['file']['tmp_name'], $targetFile)) {
                            $errors[] = $errorMessages['upload_failed'][$lang];
                            error_log("Upload mislukt: Kan bestand niet verplaatsen naar $targetFile");
                        } else {
                            if (file_exists($slides[$index]['file'])) unlink($slides[$index]['file']);
                            $slide['file'] = $targetFile;
                            error_log("Bestand geüpload: $targetFile");
                        }
                    }
                } elseif ($_FILES['file']['error'] === UPLOAD_ERR_INI_SIZE) {
                    $errors[] = $errorMessages['file_size_limit'][$lang];
                } elseif ($_FILES['file']['error'] !== UPLOAD_ERR_NO_FILE) {
                    $errors[] = sprintf($errorMessages['upload_error'][$lang], $_FILES['file']['error']);
                } elseif (isset($_POST['remove_file']) && $_POST['remove_file'] === '1') {
                    if (file_exists($slides[$index]['file'])) unlink($slides[$index]['file']);
                    $slide['file'] = '';
                }
                if (isset($_FILES['logo']) && $_FILES['logo']['error'] === UPLOAD_ERR_OK) {
                    if (!validateFileType($_FILES['logo'], 'logo')) {
                        $errors[] = $errorMessages['invalid_file_type'][$lang]['logo'];
                    } elseif ($_FILES['logo']['size'] > 50000000) {
                        $errors[] = $errorMessages['logo_too_large'][$lang];
                    } else {
                        $uniqueName = generateUniqueFilename($_FILES['logo']['name']);
                        $targetFile = $logosDir . $uniqueName;
                        if (!move_uploaded_file($_FILES['logo']['tmp_name'], $targetFile)) {
                            $errors[] = $errorMessages['logo_upload_failed'][$lang];
                            error_log("Logo upload mislukt: Kan bestand niet verplaatsen naar $targetFile");
                        } else {
                            if (isset($slides[$index]['logo']) && file_exists($slides[$index]['logo'])) unlink($slides[$index]['logo']);
                            $slide['logo'] = $targetFile;
                            error_log("Logo geüpload: $targetFile");
                        }
                    }
                } elseif (isset($_POST['remove_logo']) && $_POST['remove_logo'] === '1') {
                    if (isset($slides[$index]['logo']) && file_exists($slides[$index]['logo'])) unlink($slides[$index]['logo']);
                    $slide['logo'] = '';
                } elseif ($_FILES['logo']['error'] === UPLOAD_ERR_INI_SIZE) {
                    $errors[] = $errorMessages['logo_size_limit'][$lang];
                } elseif ($_FILES['logo']['error'] !== UPLOAD_ERR_NO_FILE) {
                    $errors[] = sprintf($errorMessages['logo_upload_error'][$lang], $_FILES['logo']['error']);
                }
                if (empty($errors)) {
                    $slides[$index] = $slide;
                    error_log("Slide bijgewerkt: " . print_r($slide, true));
                }
            }
        }
    } elseif ($action === 'delete') {
        $index = (int)$_POST['index'];
        if (isset($slides[$index])) {
            if (file_exists($slides[$index]['file'])) unlink($slides[$index]['file']);
            if (!empty($slides[$index]['logo']) && file_exists($slides[$index]['logo'])) unlink($slides[$index]['logo']);
            unset($slides[$index]);
            $slides = array_values($slides);
            error_log("Slide verwijderd: index $index");
        } else {
            $errors[] = $errorMessages['delete_failed'][$lang];
            error_log("Delete mislukt: Ongeldige index $index");
        }
    } elseif ($action === 'move_up' || $action === 'move_down') {
        $index = (int)$_POST['index'];
        if ($action === 'move_up' && $index > 0 && isset($slides[$index])) {
            $temp = $slides[$index - 1];
            $slides[$index - 1] = $slides[$index];
            $slides[$index] = $temp;
            error_log("Move up: Slide at index $index moved to " . ($index - 1));
        } elseif ($action === 'move_down' && $index < count($slides) - 1 && isset($slides[$index])) {
            $temp = $slides[$index + 1];
            $slides[$index + 1] = $slides[$index];
            $slides[$index] = $temp;
            error_log("Move down: Slide at index $index moved to " . ($index + 1));
        } else {
            $errors[] = $errorMessages['move_failed'][$lang];
            error_log("Move mislukt: Ongeldige actie $action of index $index");
        }
    }
    if (empty($errors)) {
        if ($error = saveSlides($slides, $slidesFile, $lang)) {
            $errors[] = $error;
        } else {
            header('Location: index.php?lang=' . $lang);
            exit;
        }
    }
}

// Nederlands: Bepaal de status van een slide
// English: Determine the status of a slide
function getSlideStatus($slide) {
    $now = new DateTime();
    $start = DateTime::createFromFormat('Y-m-d H:i', $slide['start']);
    $end = DateTime::createFromFormat('Y-m-d H:i', $slide['end']);
    if ($now > $end) return ['class' => 'light-red', 'text' => ['nl' => 'Verlopen', 'en' => 'Expired']];
    if ($now < $start) return ['class' => 'light-blue', 'text' => ['nl' => 'Toekomst', 'en' => 'Future']];
    return ['class' => 'light-green', 'text' => ['nl' => 'Actief', 'en' => 'Active']];
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>De Beeldkrant</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <style>
        body {
            background-color: #FFC107; /* Dark yellow background */
        }
        body::before {
            content: '';
            position: absolute;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            background: rgba(255, 255, 255, 0.6);
            z-index: -1;
        }
        .preview-container {
            position: relative;
        }
        .preview-container img, .preview-container video {
            width: 100%;
            max-width: 150px; /* Limit preview size in table */
            height: auto;
            object-fit: contain;
        }
        .preview-container .logo-preview {
            position: absolute;
            top: 16px; /* Further downward */
            left: 16px; /* Further to the right */
            width: 25%; /* Proportional to preview width */
            max-width: 80px; /* Cap for larger previews */
            height: auto;
            object-fit: contain; /* Maintain aspect ratio */
        }
        #edit-preview {
            max-width: 300px; /* Larger preview during editing */
        }
        #edit-preview img, #edit-preview video {
            width: 100%;
            height: auto;
            object-fit: contain;
        }
        #edit-preview .logo-preview {
            position: absolute;
            top: 16px; /* Consistent positioning */
            left: 16px; /* Consistent positioning */
            width: 25%; /* Proportional to preview width */
            max-width: 80px; /* Cap for larger previews */
            height: auto;
            object-fit: contain; /* Maintain aspect ratio */
        }
        #previewModal {
            display: none;
            position: fixed;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background: rgba(0, 0, 0, 0.5);
            z-index: 1000;
            justify-content: center;
            align-items: center;
        }
        #previewModal > div {
            background: #E6F0FA; /* Pastel blue for modal */
            padding: 20px;
            width: 80%;
            max-height: 80%;
            border-radius: 8px;
            overflow: auto;
        }
        #previewModal img, #previewModal video {
            width: 100%;
            height: auto;
            object-fit: contain;
        }
        #previewModal .logo-preview {
            position: absolute;
            top: 16px; /* Consistent positioning */
            left: 16px; /* Consistent positioning */
            width: 25%; /* Proportional to preview width */
            max-width: 80px; /* Cap for larger previews */
            height: auto;
            object-fit: contain; /* Maintain aspect ratio */
        }
    </style>
</head>
<body class="font-sans text-gray-800 min-h-screen relative">
    <img id="admin-logo" src="logo_thuis_wageningen.webp" alt="Logo" class="absolute top-4 left-4 w-32 z-10">
    <div class="container mx-auto px-4 py-8 max-w-6xl">
        <div class="flex justify-center items-center mb-6">
            <select id="lang-select" onchange="toggleLanguage()" class="p-2 border border-gray-300 rounded bg-gray-50 focus:outline-none focus:border-teal-400 focus:ring-2 focus:ring-teal-200">
                <option value="nl" data-lang-nl="Nederlands" data-lang-en="Dutch">Nederlands</option>
                <option value="en" data-lang-nl="Engels" data-lang-en="English">Engels</option>
            </select>
        </div>
        <h1 class="text-4xl font-bold text-teal-600 text-center mb-8" data-lang-nl="De Beeldkrant" data-lang-en="De Beeldkrant">De Beeldkrant</h1>
        <?php if (!empty($errors)): ?>
            <div class="bg-pink-100 text-pink-600 p-4 rounded-lg shadow-lg mb-8 text-center">
                <?php foreach ($errors as $error): ?>
                    <p><?php echo htmlspecialchars($error); ?></p>
                <?php endforeach; ?>
            </div>
        <?php endif; ?>
        <div id="slide-form" class="bg-blue-100 p-6 rounded-lg shadow-lg mb-8">
            <h2 id="form-title" class="text-2xl font-semibold text-teal-600 mb-4" data-lang-nl="Nieuwe Slide Toevoegen" data-lang-en="Add New Slide">Nieuwe Slide Toevoegen</h2>
            <form method="POST" enctype="multipart/form-data" class="grid grid-cols-1 md:grid-cols-2 gap-6">
                <input type="hidden" name="action" value="add">
                <input type="hidden" name="index" value="">
                <input type="hidden" name="lang" value="<?php echo isset($_GET['lang']) && $_GET['lang'] === 'en' ? 'en' : 'nl'; ?>">
                <label class="block" data-lang-nl="Naam:" data-lang-en="Name:">Naam:
                    <input type="text" name="name" required class="mt-1 w-full p-2 border border-gray-300 rounded bg-gray-50 focus:outline-none focus:border-teal-400 focus:ring-2 focus:ring-teal-200">
                </label>
                <label class="block" data-lang-nl="Type:" data-lang-en="Type:">Type:
                    <select name="type" required onchange="toggleDuration(); updateEditPreview();" class="mt-1 w-full p-2 border border-gray-300 rounded bg-gray-50 focus:outline-none focus:border-teal-400 focus:ring-2 focus:ring-teal-200">
                        <option value="image" data-lang-nl="Afbeelding" data-lang-en="Image">Afbeelding</option>
                        <option value="video" data-lang-nl="Video" data-lang-en="Video">Video</option>
                    </select>
                </label>
                <label class="block" data-lang-nl="Bestand:" data-lang-en="File:">Bestand:
                    <input type="file" name="file" class="mt-1 w-full p-2 border border-gray-300 rounded bg-gray-50" onchange="updateEditPreview()">
                </label>
                <label class="block" data-lang-nl="Logo:" data-lang-en="Logo:">Logo:
                    <input type="file" name="logo" class="mt-1 w-full p-2 border border-gray-300 rounded bg-gray-50" onchange="updateEditPreview()">
                </label>
                <label id="remove_file_label" class="block hidden" data-lang-nl="Bestand verwijderen:" data-lang-en="Remove file:">Bestand verwijderen:
                    <input type="checkbox" name="remove_file" value="1" class="mt-1" onchange="updateEditPreview()">
                </label>
                <label id="remove_logo_label" class="block hidden" data-lang-nl="Logo verwijderen:" data-lang-en="Remove logo:">Logo verwijderen:
                    <input type="checkbox" name="remove_logo" value="1" class="mt-1" onchange="updateEditPreview()">
                </label>
                <label class="block" data-lang-nl="Start (YYYY-MM-DD HH:MM):" data-lang-en="Start (YYYY-MM-DD HH:MM):">Start (YYYY-MM-DD HH:MM):
                    <input type="text" name="start" required placeholder="2025-12-31 14:00" class="mt-1 w-full p-2 border border-gray-300 rounded bg-gray-50 focus:outline-none focus:border-teal-400 focus:ring-2 focus:ring-teal-200">
                </label>
                <label class="block" data-lang-nl="Eind (YYYY-MM-DD HH:MM):" data-lang-en="End (YYYY-MM-DD HH:MM):">Eind (YYYY-MM-DD HH:MM):
                    <input type="text" name="end" required placeholder="2025-12-31 15:00" class="mt-1 w-full p-2 border border-gray-300 rounded bg-gray-50 focus:outline-none focus:border-teal-400 focus:ring-2 focus:ring-teal-200">
                </label>
                <label class="block" data-lang-nl="Herhaling:" data-lang-en="Repeat:">Herhaling:
                    <select name="repeat_type" class="mt-1 w-full p-2 border border-gray-300 rounded bg-gray-50 focus:outline-none focus:border-teal-400 focus:ring-2 focus:ring-teal-200">
                        <option value="none" data-lang-nl="Geen" data-lang-en="None">Geen</option>
                        <option value="daily" data-lang-nl="Dagelijks" data-lang-en="Daily">Dagelijks</option>
                        <option value="every_x_days" data-lang-nl="Om de X dagen" data-lang-en="Every X Days">Om de X dagen</option>
                        <option value="weekly" data-lang-nl="Wekelijks" data-lang-en="Weekly">Wekelijks</option>
                        <option value="every_x_weeks" data-lang-nl="Om de X weken" data-lang-en="Every X Weeks">Om de X weken</option>
                        <option value="monthly" data-lang-nl="Maandelijks" data-lang-en="Monthly">Maandelijks</option>
                        <option value="every_x_months" data-lang-nl="Om de X maanden" data-lang-en="Every X Months">Om de X maanden</option>
                        <option value="yearly" data-lang-nl="Jaarlijks" data-lang-en="Yearly">Jaarlijks</option>
                        <option value="every_x_years" data-lang-nl="Om de X jaren" data-lang-en="Every X Years">Om de X jaren</option>
                    </select>
                </label>
                <label class="block" data-lang-nl="Interval:" data-lang-en="Interval:">Interval:
                    <input type="number" name="interval" value="1" min="1" required class="mt-1 w-full p-2 border border-gray-300 rounded bg-gray-50 focus:outline-none focus:border-teal-400 focus:ring-2 focus:ring-teal-200">
                </label>
                <label id="duration_label" class="block" data-lang-nl="Duurtijd (ms, alleen voor afbeeldingen):" data-lang-en="Duration (ms, images only):">Duurtijd (ms, alleen voor afbeeldingen):
                    <input type="number" name="duration" value="10000" min="1000" class="mt-1 w-full p-2 border border-gray-300 rounded bg-gray-50 focus:outline-none focus:border-teal-400 focus:ring-2 focus:ring-teal-200">
                </label>
                <div class="col-span-2 flex justify-center space-x-4">
                    <button type="submit" id="submit-button" class="bg-teal-400 text-white px-6 py-2 rounded hover:bg-teal-500 transition disabled:bg-gray-400 disabled:cursor-not-allowed" data-lang-nl-add="Toevoegen" data-lang-en-add="Add" data-lang-nl-edit="Updaten" data-lang-en-edit="Update">Toevoegen</button>
                    <button type="button" onclick="resetForm()" class="bg-pink-400 text-white px-6 py-2 rounded hover:bg-pink-500 transition" data-lang-nl="Annuleren" data-lang-en="Cancel">Annuleren</button>
                </div>
                <div id="edit-preview" class="col-span-2 preview-container hidden mt-4"></div>
            </form>
        </div>
        <table class="w-full bg-blue-100 shadow-lg rounded-lg overflow-hidden">
            <thead>
                <tr class="bg-teal-400 text-white">
                    <th class="p-4" data-lang-nl="Naam" data-lang-en="Name">Naam</th>
                    <th class="p-4" data-lang-nl="Type" data-lang-en="Type">Type</th>
                    <th class="p-4" data-lang-nl="Start" data-lang-en="Start">Start</th>
                    <th class="p-4" data-lang-nl="Eind" data-lang-en="End">Eind</th>
                    <th class="p-4" data-lang-nl="Herhaling" data-lang-en="Repeat">Herhaling</th>
                    <th class="p-4" data-lang-nl="Interval" data-lang-en="Interval">Interval</th>
                    <th class="p-4" data-lang-nl="Status" data-lang-en="Status">Status</th>
                    <th class="p-4" data-lang-nl="Preview" data-lang-en="Preview">Preview</th>
                    <th class="p-4" data-lang-nl="Acties" data-lang-en="Actions">Acties</th>
                </tr>
            </thead>
            <tbody>
                <?php
                $lang = isset($_GET['lang']) && $_GET['lang'] === 'en' ? 'en' : 'nl';
                ?>
                <?php if (empty($slides)): ?>
                    <tr><td colspan="9" class="p-4 text-center" data-lang-nl="Geen slides beschikbaar." data-lang-en="No slides available."><?php echo $lang === 'nl' ? 'Geen slides beschikbaar.' : 'No slides available.'; ?></td></tr>
                <?php else: ?>
                    <?php foreach ($slides as $index => $slide): ?>
                        <?php if (!isset($slide['name'], $slide['type'], $slide['start'], $slide['end'], $slide['repeat_type'], $slide['interval'], $slide['file'])) continue;
                        $status = getSlideStatus($slide); ?>
                        <tr class="border-b-2 border-gray-300">
                            <td class="p-4"><?php echo htmlspecialchars($slide['name']); ?></td>
                            <td class="p-4" data-lang-nl="<?php echo htmlspecialchars($slide['type'] === 'image' ? 'Afbeelding' : 'Video'); ?>" data-lang-en="<?php echo htmlspecialchars($slide['type']); ?>">
                                <?php echo htmlspecialchars($slide['type'] === 'image' ? ($lang === 'nl' ? 'Afbeelding' : 'Image') : ($lang === 'nl' ? 'Video' : 'Video')); ?>
                            </td>
                            <td class="p-4"><?php echo htmlspecialchars($slide['start']); ?></td>
                            <td class="p-4"><?php echo htmlspecialchars($slide['end']); ?></td>
                            <td class="p-4" data-lang-nl="<?php echo htmlspecialchars($slide['repeat_type'] === 'none' ? 'Geen' : $slide['repeat_type']); ?>" data-lang-en="<?php echo htmlspecialchars($slide['repeat_type'] === 'none' ? 'None' : $slide['repeat_type']); ?>">
                                <?php echo htmlspecialchars($slide['repeat_type'] === 'none' ? ($lang === 'nl' ? 'Geen' : 'None') : $slide['repeat_type']); ?>
                            </td>
                            <td class="p-4"><?php echo htmlspecialchars($slide['interval']); ?></td>
                            <td class="p-4"><span class="text-<?php echo $status['class'] === 'light-red' ? 'pink-600' : ($status['class'] === 'light-blue' ? 'blue-600' : 'teal-600'); ?>" data-lang-nl="<?php echo $status['text']['nl']; ?>" data-lang-en="<?php echo $status['text']['en']; ?>"><?php echo $lang === 'nl' ? $status['text']['nl'] : $status['text']['en']; ?></span></td>
                            <td class="p-4">
                                <div class="preview-container relative">
                                    <?php if ($slide['type'] === 'image'): ?>
                                        <img src="<?php echo htmlspecialchars($slide['file']); ?>" alt="Preview" onerror="console.error('Failed to load image: <?php echo htmlspecialchars($slide['file']); ?>')">
                                    <?php else: ?>
                                        <video src="<?php echo htmlspecialchars($slide['file']); ?>" autoplay loop muted onloadeddata="this.play().catch(e => console.error('Video play error:', e))" onerror="console.error('Failed to load video: <?php echo htmlspecialchars($slide['file']); ?>')"></video>
                                    <?php endif; ?>
                                    <?php if (!empty($slide['logo'])): ?>
                                        <img src="<?php echo htmlspecialchars($slide['logo']); ?>" class="logo-preview" onerror="console.error('Failed to load logo: <?php echo htmlspecialchars($slide['logo']); ?>')">
                                    <?php endif; ?>
                                </div>
                            </td>
                            <td class="p-4 flex space-x-2">
                                <button type="button" onclick="editSlide(<?php echo $index; ?>)" class="bg-blue-400 text-white px-3 py-1 rounded hover:bg-blue-500" data-lang-nl="Bewerken" data-lang-en="Edit"><?php echo $lang === 'nl' ? 'Bewerken' : 'Edit'; ?></button>
                                <button type="button" onclick="previewSlide(<?php echo $index; ?>)" class="bg-gray-400 text-white px-3 py-1 rounded hover:bg-gray-500" data-lang-nl="Preview" data-lang-en="Preview"><?php echo $lang === 'nl' ? 'Preview' : 'Preview'; ?></button>
                                <form method="POST" class="inline"><input type="hidden" name="action" value="delete"><input type="hidden" name="index" value="<?php echo $index; ?>"><input type="hidden" name="lang" value="<?php echo $lang; ?>"><button type="submit" class="bg-pink-400 text-white px-3 py-1 rounded hover:bg-pink-500" data-lang-nl="Verwijderen" data-lang-en="Delete"><?php echo $lang === 'nl' ? 'Verwijderen' : 'Delete'; ?></button></form>
                                <form method="POST" class="inline"><input type="hidden" name="action" value="move_up"><input type="hidden" name="index" value="<?php echo $index; ?>"><input type="hidden" name="lang" value="<?php echo $lang; ?>"><button type="submit" class="bg-gray-400 text-white px-3 py-1 rounded hover:bg-gray-500" <?php echo $index === 0 ? 'disabled' : ''; ?>>↑</button></form>
                                <form method="POST" class="inline"><input type="hidden" name="action" value="move_down"><input type="hidden" name="index" value="<?php echo $index; ?>"><input type="hidden" name="lang" value="<?php echo $lang; ?>"><button type="submit" class="bg-gray-400 text-white px-3 py-1 rounded hover:bg-gray-500" <?php echo $index === count($slides) - 1 ? 'disabled' : ''; ?>>↓</button></form>
                            </td>
                        </tr>
                    <?php endforeach; ?>
                <?php endif; ?>
            </tbody>
        </table>
        <div id="previewModal" class="hidden">
            <div class="relative">
                <div id="previewContent"></div>
                <button class="bg-gray-400 text-white px-6 py-2 rounded hover:bg-gray-500 mt-4" data-lang-nl="Sluiten" data-lang-en="Close" onclick="document.getElementById('previewModal').style.display='none'">Sluiten</button>
            </div>
        </div>
    </div>
    <script>
        const slides = <?php echo json_encode($slides); ?>;
        const langElements = document.querySelectorAll('[data-lang-nl],[data-lang-en]');
        const formTitle = document.querySelector('#form-title');
        const submitButton = document.querySelector('#submit-button');
        const form = document.querySelector('#slide-form form');
        const editPreview = document.getElementById('edit-preview');
        const fileInput = document.querySelector('input[name="file"]');
        const logoInput = document.querySelector('input[name="logo"]');
        const removeFileCheckbox = document.querySelector('input[name="remove_file"]');
        const removeLogoCheckbox = document.querySelector('input[name="remove_logo"]');

        function toggleDuration() {
            document.getElementById('duration_label').classList.toggle('hidden', document.querySelector('[name="type"]').value !== 'image');
        }

        function resetForm() {
            form.reset();
            formTitle.textContent = getLangText('form-title');
            document.querySelector('[name="action"]').value = 'add';
            document.querySelector('[name="index"]').value = '';
            editPreview.classList.add('hidden');
            document.getElementById('remove_file_label').classList.add('hidden');
            document.getElementById('remove_logo_label').classList.add('hidden');
            submitButton.textContent = localStorage.getItem('language') === 'en' ? submitButton.getAttribute('data-lang-en-add') : submitButton.getAttribute('data-lang-nl-add');
            toggleDuration();
        }

        function updateEditPreview() {
            const index = document.querySelector('[name="index"]').value;
            const slide = index ? slides[index] : null;
            const type = document.querySelector('[name="type"]').value;
            let previewContent = '';
            if (slide && slide.file && !removeFileCheckbox.checked) {
                previewContent = type === 'image' ? `<img src="${slide.file}" class="w-full h-auto object-contain" onerror="console.error('Failed to load edit preview image: ${slide.file}')">` : `<video src="${slide.file}" autoplay loop muted class="w-full h-auto object-contain" onloadeddata="this.play().catch(e => console.error('Video play error:', e))" onerror="console.error('Failed to load edit preview video: ${slide.file}')"></video>`;
            }
            const file = fileInput.files[0];
            if (file) {
                previewContent = type === 'image' ? `<img src="${URL.createObjectURL(file)}" class="w-full h-auto object-contain" onerror="console.error('Failed to load new file preview')">` : `<video src="${URL.createObjectURL(file)}" autoplay loop muted class="w-full h-auto object-contain" onloadeddata="this.play().catch(e => console.error('Video play error:', e))" onerror="console.error('Failed to load new video preview')"></video>`;
            }
            const logoFile = logoInput.files[0];
            const showLogo = slide && slide.logo && !removeLogoCheckbox.checked;
            if (logoFile) {
                previewContent += `<img src="${URL.createObjectURL(logoFile)}" class="logo-preview" onerror="console.error('Failed to load new logo preview')">`;
            } else if (showLogo) {
                previewContent += `<img src="${slide.logo}" class="logo-preview" onerror="console.error('Failed to load logo preview: ${slide.logo}')">`;
            }
            editPreview.innerHTML = previewContent;
            editPreview.classList.toggle('hidden', !previewContent);
            if (!previewContent) console.warn('Edit preview is empty');
        }

        function editSlide(index) {
            const slide = slides[index];
            if (!slide) {
                console.error(`Slide at index ${index} not found`);
                return;
            }
            formTitle.textContent = localStorage.getItem('language') === 'en' ? formTitle.getAttribute('data-lang-en').replace('Add', 'Edit') : formTitle.getAttribute('data-lang-nl').replace('Toevoegen', 'Bewerken');
            document.querySelector('[name="action"]').value = 'edit';
            document.querySelector('[name="index"]').value = index;
            document.querySelector('[name="name"]').value = slide.name || '';
            document.querySelector('[name="type"]').value = slide.type || 'image';
            document.querySelector('[name="start"]').value = slide.start || '';
            document.querySelector('[name="end"]').value = slide.end || '';
            document.querySelector('[name="repeat_type"]').value = slide.repeat_type || 'none';
            document.querySelector('[name="interval"]').value = slide.interval || 1;
            if (slide.type === 'image') document.querySelector('[name="duration"]').value = slide.duration || 10000;
            document.getElementById('remove_file_label').classList.toggle('hidden', !slide.file);
            document.getElementById('remove_logo_label').classList.toggle('hidden', !slide.logo);
            submitButton.textContent = localStorage.getItem('language') === 'en' ? submitButton.getAttribute('data-lang-en-edit') : submitButton.getAttribute('data-lang-nl-edit');
            updateEditPreview();
            toggleDuration();
        }

        function previewSlide(index) {
            const slide = slides[index];
            if (!slide) {
                console.error(`Slide at index ${index} not found`);
                return;
            }
            let content = slide.type === 'image' ? `<img src="${slide.file}" class="w-full h-full object-contain" onerror="console.error('Failed to load modal preview image: ${slide.file}')">` : `<video src="${slide.file}" autoplay loop muted class="w-full h-full object-contain" onloadeddata="this.play().catch(e => console.error('Video play error:', e))" onerror="console.error('Failed to load modal preview video: ${slide.file}')"></video>`;
            if (slide.logo) {
                content += `<img src="${slide.logo}" class="logo-preview" onerror="console.error('Failed to load modal logo: ${slide.logo}')">`;
            }
            const previewContent = document.getElementById('previewContent');
            previewContent.innerHTML = content;
            document.getElementById('previewModal').style.display = 'flex';
            console.log(`Previewing slide ${index}: ${slide.file}${slide.logo ? ', logo: ' + slide.logo : ''}`);
        }

        function getLangText(elementId) {
            const lang = localStorage.getItem('language') || 'nl';
            const element = document.querySelector(`#${elementId},[data-lang-${lang}]`);
            return element?.getAttribute(`data-lang-${lang}`) || '';
        }

        function getLangTextNone() {
            return localStorage.getItem('language') === 'en' ? 'None' : 'Geen';
        }

        function toggleLanguage() {
            const lang = document.getElementById('lang-select').value;
            localStorage.setItem('language', lang);
            langElements.forEach(elem => {
                const text = elem.getAttribute(`data-lang-${lang}`);
                if (elem.tagName === 'LABEL' || elem.tagName === 'TH' || (elem.tagName === 'TD' && elem.hasAttribute(`data-lang-${lang}`))) {
                    elem.childNodes[0].nodeValue = text;
                } else {
                    elem.textContent = text;
                }
            });
            document.querySelector('title').textContent = getLangText('title');
            formTitle.textContent = document.querySelector('[name="action"]').value === 'edit' 
                ? (lang === 'en' ? formTitle.getAttribute('data-lang-en').replace('Add', 'Edit') : formTitle.getAttribute('data-lang-nl').replace('Toevoegen', 'Bewerken'))
                : getLangText('form-title');
            submitButton.textContent = document.querySelector('[name="action"]').value === 'edit' 
                ? (lang === 'en' ? submitButton.getAttribute('data-lang-en-edit') : submitButton.getAttribute('data-lang-nl-edit'))
                : (lang === 'en' ? submitButton.getAttribute('data-lang-en-add') : submitButton.getAttribute('data-lang-nl-add'));
            document.querySelector('input[name="lang"]').value = lang;
        }

        if (localStorage.getItem('language') === 'en') {
            document.getElementById('lang-select').value = 'en';
            toggleLanguage();
        }
        toggleDuration();
    </script>
</body>
</html>